////////////////////////////////////////////////////////////////////////////////
// Filename: depth.hlsli
////////////////////////////////////////////////////////////////////////////////

/////////////
// GLOBALS //
/////////////

cbuffer MatrixBuffer : register(b0) {
    matrix worldMatrix;               // Transform into world space
};

//////////////////////
// CONSTANT BUFFERS //
//////////////////////

struct CameraType {
    matrix view;                      // Camera view matrix
    matrix projection;                // Transform world to view space and project

	float4 position;                  // Camera position (eye)
	float4 target;                    // Camera target (lookAt)
    float4 direction;                // World space light direction
	float4 rotation;                  // Camera rotation

	float2 plane;                     // Clipping plane (near/far)
	float  FOV;                       // Field of view (radians)
    float  alpha;                     // View transparency (for fades)
};

cbuffer CameraBuffer : register(b1)
{
    CameraType camera;
};

struct LightType : CameraType {
    float4  colour;                   // Light color/intensity (RGBA)
    float2  linearNearFar;            // Distances to use for rescaling shadows
    float2  distanceFalloff;          // Light distance falloff
    float2  angleFalloff;             // Light angular falloff (relative to FOV)
};

cbuffer LightBuffer : register(b2) {
    LightType light;
};

//--------------------------------------------------------------------------------------
// Misc Utilities
//--------------------------------------------------------------------------------------

float scale(float min, float max, float value) {
    return clamp((value - min) / (max - min), 0, 1);
}

float2 bias() {
    return float2(0.5, 0);
}

float2 moments(float depth)
{
    // Compute first few moments of depth
    return float2(depth, depth * depth);
}


//////////////
// TYPEDEFS //
//////////////

struct VertexInput {
    float4 position : POSITION; // Object space position
    float4 colour : COLOR;      // Object colour
    float3 normal : NORMAL;     // Object space normal
};

struct PixelInput {
    float4 position : SV_POSITION;
    float3 depthPosition : TEXTURE0;
    //float4 viewPosition : TEXCOORD;
};

////////////////////////////////////////////////////////////////////////////////
// Vertex Shader
////////////////////////////////////////////////////////////////////////////////
PixelInput vertexShader(VertexInput input) {
    PixelInput output;
    input.position.w = 1.0f;

    // Calculate the position of the vertex against the view and projection matrices.
    output.position = mul(input.position, camera.projection);
    output.depthPosition = mul(input.position, camera.view).xyz;
	
    return output;
}

////////////////////////////////////////////////////////////////////////////////
// Pixel Shader
////////////////////////////////////////////////////////////////////////////////

float2 pixelShader(centroid PixelInput input) : SV_TARGET{
    //float depth = input.depthPosition.z / input.depthPosition.w;
    //depth = 1.f - 0.01 / (1.01 - depth); // linearise if visualising

    float2 moments;
    moments.x = scale(light.linearNearFar.x, light.linearNearFar.y, length(input.depthPosition));

    const float dx = ddx(moments.x);
    const float dy = ddy(moments.x);
    moments.y = moments.x * moments.x +0.5 * (dx * dx + dy * dy);

    return moments;
}